using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Negotiator.BusinessObjects;
using System.Xml;
using System.Xml.Serialization;

namespace Negotiator.Svc
{
    public class Service
    {
        const decimal outrageousLimit = 0.75M;
        public bool Start()
        {
            Console.WriteLine("Service started...");

            /*get the supplier limits to enforce*/
            var supplierLimits = BOSupplierLimits.SupplierLimitsCollection();

            while(true)
            {
                /*get all new orders*/
                int supplierId;
                var newOrders = new Criteria<BOPendingOrders>().Add(Expression.Eq("OrderStatus", "NEW")).List<BOPendingOrders>();
                foreach(var order in newOrders)
                {
                    if(IsOutrageous(order, supplierLimits))
                    {
                        Console.WriteLine("OrderId " + order.OrderId + " is outrageous, and will be rejected outright");
                        RejectOrder(order, "Outrageous");
                    }
                    else if(HasExceededPatience(order, supplierLimits))
                    {
                        Console.WriteLine("OrderId " + order.OrderId + " exceeded patience at bid number " + order.BidCount + ", rejected");
                        RejectOrder(order, "Has exceeded patience at bid number " + order.BidCount);
                    }
                    else if(IsNotQuiteThere(order, supplierLimits))
                    {
                        Console.WriteLine("OrderId " + order.OrderId + " is not quite there, courteously declined");
                        DeclineOrder(order);
                    }
                    else if(IsInTheMoney(order, supplierLimits, out supplierId))
                    {
                        Console.WriteLine("OrderId " + order.OrderId + " is in the money, and will be agreed");
                        AgreeOrder(order, supplierId);
                    }
                }
            }
        }

        private bool IsOutrageous(BOPendingOrders pendingOrder, IList<BOSupplierLimits> supplierLimits)
        {
            if (pendingOrder.BidPrice < supplierLimits.Where((x)=>x.ProductId == pendingOrder.ProductId).Min((x) => x.MinBid) * outrageousLimit)
                return true;
            return false;
        }

        private bool HasExceededPatience(BOPendingOrders pendingOrder, IList<BOSupplierLimits> supplierLimits)
        {
            if (pendingOrder.BidCount > supplierLimits.Where((x) => x.ProductId == pendingOrder.ProductId).Max((x) => x.MaxBidsPerOrder))
                return true;
            return false;
        }

        private bool IsNotQuiteThere(BOPendingOrders pendingOrder, IList<BOSupplierLimits> supplierLimits)
        {
            if (pendingOrder.BidPrice < supplierLimits.Where((x) => x.ProductId == pendingOrder.ProductId).Min((x) => x.MinBid))
                return true;
            return false;
        }

        private bool IsInTheMoney(BOPendingOrders pendingOrder, IList<BOSupplierLimits> supplierLimits, out int supplierId)
        {
            var supplierLimit = supplierLimits.Where((x) => x.ProductId == pendingOrder.ProductId)
                .FirstOrDefault((x) => x.MinBid <= pendingOrder.BidPrice);

            supplierId = supplierLimit != null ? (int)supplierLimit.SupplierId : -1;
            return supplierId != -1;
        }

        private void AgreeOrder(BOPendingOrders pendingOrder, int supplierId)
        {
            var agreed = new BOAgreedOrders();
            agreed.OrderDetails = Serialize(pendingOrder);
            agreed.CustomerId = pendingOrder.CustomerId;
            agreed.SupplierId = supplierId;
            agreed.Status = "AGREED";

            /*should really use a unit of work transaction here. see CodeTrigger UnitOfWork samples*/
            agreed.SaveNew();
            pendingOrder.Delete();
            /***************************************************************************************/
        }

        private void DeclineOrder(BOPendingOrders pendingOrder)
        {
            pendingOrder.OrderStatus = "DECLINED";
            pendingOrder.Update();
        }

        private void RejectOrder(BOPendingOrders pendingOrder, string reason)
        {
            var rejected = new BORejectedOrders();
            rejected.OrderDetails = Serialize(pendingOrder);
            rejected.Reason = reason;

            /*should really use a unit of work transaction here. see CodeTrigger UnitOfWork samples*/
            rejected.SaveNew();
            pendingOrder.Delete();
            /***************************************************************************************/
        }

        private string Serialize(BOPendingOrders pendingOrder)
        {
            XmlSerializer serializer = new XmlSerializer(typeof(BOPendingOrders));
            StringBuilder result = new StringBuilder();
            using (var writer = XmlWriter.Create(result))
            {
                serializer.Serialize(writer, pendingOrder);
            }
            return result.ToString();
        }

        public bool Stop()
        {
            return true;
        }

        public bool Pause()
        {
            Console.WriteLine("Service paused...");
            return true;
        }

        public bool Continue()
        {
            Console.WriteLine("Service continued...");
            return true;
        }
    }
}
